<?php

namespace Tests\Feature;

use App\Models\FeeStructure;
use App\Models\Payment;
use App\Models\Subscription;
use App\Models\User;
use App\Services\FeeCalculatorService;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class FinanceIntegrationTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_can_manage_fee_structures()
    {
        $admin = User::factory()->create(['email' => 'admin@finance.com']);
        $this->actingAs($admin);

        // Create
        $response = $this->post(route('admin.finance.fee-structures.store'), [
            'name' => 'Monthly Standard',
            'amount' => 100.00,
            'frequency' => 'monthly',
        ]);
        $response->assertRedirect(route('admin.finance.fee-structures.index'));
        $this->assertDatabaseHas('fee_structures', ['name' => 'Monthly Standard']);

        // List
        $this->get(route('admin.finance.fee-structures.index'))->assertSee('Monthly Standard');
    }

    public function test_balance_calculation_logic()
    {
        $service = new FeeCalculatorService();
        $user = User::factory()->create();
        
        // Create Fee Structure: Annual 1200
        $structure = FeeStructure::create([
            'name' => 'Annual',
            'amount' => 1200,
            'frequency' => 'annual',
        ]);

        // Subscribe user starting 1 year ago (so 2 periods: last year and this year? 
        // Logic: Annual charge is usually up-front. If start date is 1 year ago, 1st charge.
        // If end date is now, diffInYears = 1. +1 = 2 charges? 
        // Let's test Monthly for clearer granularity.
        
        $monthly = FeeStructure::create([
            'name' => 'Monthly',
            'amount' => 100,
            'frequency' => 'monthly',
        ]);

        // Subscribe from 3 months ago
        Subscription::create([
            'user_id' => $user->id,
            'fee_structure_id' => $monthly->id,
            'start_date' => Carbon::now()->subMonths(3), // e.g. Oct, Nov, Dec, Jan...
            // difference in months between (now-3m) and now is 3. +1 = 4 charges (Oct, Nov, Dec, Jan)
        ]);
        
        // Expected: 4 * 100 = 400
        $expected = $service->calculateExpected($user);
        // diffInMonths(now) is exactly 3? 
        // e.g. 1st Jan to 1st April = 3 months. April is 4th month.
        // Let's rely on the service implementation: $startDate->diffInMonths($endDate) + 1
        
        // payments = 0
        $balance = $service->calculateBalance($user);
        $this->assertEquals($expected, $balance);
        $this->assertEquals('Arrears', $service->getStatus($user));

        // Make full payment
        Payment::create([
            'user_id' => $user->id,
            'amount' => $expected,
            'status' => 'verified',
            'date' => now(),
        ]);

        $balance = $service->calculateBalance($user);
        $this->assertEquals(0, $balance);
        $this->assertEquals('Paid Up', $service->getStatus($user));
    }

    public function test_contributions_view_shows_correct_status()
    {
        \Spatie\Permission\Models\Role::create(['name' => 'member']);

        $admin = User::factory()->create();
        $this->actingAs($admin);
        
        $member = User::factory()->create(['name' => 'John Doe']);
        $member->assignRole('member');

        $structure = FeeStructure::create(['name' => 'Test Fee', 'amount' => 50, 'frequency' => 'one-time']);
        Subscription::create([
            'user_id' => $member->id,
            'fee_structure_id' => $structure->id,
            'start_date' => now()->subDay(),
        ]);

        // Should be in arrears (50 owed, 0 paid)
        $response = $this->get(route('admin.finance.contributions'));
        $response->assertSee('John Doe');
        $response->assertSee('Arrears'); // Status
        $response->assertSee('-R 50.00'); // Balance display formatted
    }
}
