<?php

namespace Tests\Feature;

use App\Models\Document;
use App\Models\DocumentCategory;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class DocumentsTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_can_manage_categories()
    {
        $admin = User::factory()->create(['email' => 'admin@docs.com']);
        $this->actingAs($admin);

        $response = $this->post(route('admin.resources.document-categories.store'), [
            'name' => 'Policies',
            'description' => 'Official policies',
        ]);

        $this->assertDatabaseHas('document_categories', ['name' => 'Policies']);
    }

    public function test_admin_can_upload_document()
    {
        Storage::fake('public');
        $admin = User::factory()->create(['email' => 'admin@docs.com']);
        $this->actingAs($admin);

        $category = DocumentCategory::create(['name' => 'Agreements']);
        $file = UploadedFile::fake()->create('contract.pdf', 500);

        $response = $this->post(route('admin.resources.documents.store'), [
            'title' => 'Main Contract',
            'category_id' => $category->id,
            'access_level' => 'members',
            'file' => $file,
            'version_notes' => 'Initial upload',
        ]);

        // Check Document created
        $this->assertDatabaseHas('documents', ['title' => 'Main Contract']);
        
        // Check Version created
        $doc = Document::first();
        $this->assertCount(1, $doc->versions);
        $this->assertEquals('Initial upload', $doc->versions->first()->version_notes);
        
        // Check file exists
        Storage::disk('public')->assertExists($doc->versions->first()->file_path);
    }

    public function test_admin_can_upload_new_version()
    {
        Storage::fake('public');
        $admin = User::factory()->create(['email' => 'admin@docs.com']);
        $this->actingAs($admin);

        $category = DocumentCategory::create(['name' => 'Forms']);
        $doc = Document::create([
            'title' => 'Application Form',
            'category_id' => $category->id,
            'access_level' => 'public',
        ]);

        // Upload v1
        $file1 = UploadedFile::fake()->create('form_v1.pdf');
        $doc->versions()->create([
            'file_path' => 'path/to/v1',
            'file_name' => 'form_v1.pdf',
            'file_size' => 100,
            'mime_type' => 'application/pdf',
            'version_notes' => 'v1',
            'user_id' => $admin->id,
        ]);

        // Upload v2 via controller
        $file2 = UploadedFile::fake()->create('form_v2.pdf');
        $response = $this->post(route('admin.resources.documents.version.store', $doc->id), [
            'file' => $file2,
            'version_notes' => 'Updated fields',
        ]);

        $this->assertCount(2, $doc->fresh()->versions);
        $latest = $doc->fresh()->latestVersion;
        $this->assertEquals('Updated fields', $latest->version_notes);
    }
}
