<?php

namespace Tests\Feature;

use App\Models\Grievance;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class CaseIntakeTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_can_create_case_via_ui()
    {
        $admin = User::factory()->create(['email' => 'admin@test.com']);
        $member = User::factory()->create(['email' => 'member@test.com']);
        $this->actingAs($admin);

        $response = $this->post(route('admin.cases.store'), [
            'user_id' => $member->id,
            'title' => 'Unfair Dismissal',
            'category' => 'unfair_dismissal',
            'description' => 'I was fired without notice.',
            'incident_date' => '2026-01-20',
            'urgency' => 'critical',
            'employer_name' => 'Bad Corp',
        ]);

        $response->assertRedirect(route('admin.cases.index'));
        $this->assertDatabaseHas('grievances', [
            'user_id' => $member->id,
            'title' => 'Unfair Dismissal',
            'category' => 'unfair_dismissal',
            'urgency' => 'critical',
            'employer_name' => 'Bad Corp',
            'status' => 'submitted', // Default
        ]);

        // Check auto-generated reference
        $case = Grievance::first();
        $this->assertNotNull($case->reference_number);
        $this->assertStringStartsWith('CASE-' . date('Y'), $case->reference_number);
    }

    public function test_member_can_create_case_via_api()
    {
        $member = User::factory()->create();
        $this->actingAs($member);

        $response = $this->postJson('/api/cases', [
            'title' => 'Late Wages',
            'category' => 'wage_dispute',
            'description' => 'Payment late by 2 weeks.',
            'incident_date' => '2026-02-01',
            'urgency' => 'high',
            'employer_name' => 'Factory X',
        ]);

        $response->assertStatus(201)
                 ->assertJsonPath('message', 'Case submitted successfully');

        $this->assertDatabaseHas('grievances', [
            'user_id' => $member->id,
            'title' => 'Late Wages',
            'category' => 'wage_dispute',
        ]);
        
        $case = Grievance::where('title', 'Late Wages')->first();
        $this->assertNotNull($case->reference_number);
    }

    public function test_case_creation_validation()
    {
        $admin = User::factory()->create();
        $this->actingAs($admin);

        $response = $this->post(route('admin.cases.store'), []);
        $response->assertSessionHasErrors(['user_id', 'title', 'category', 'urgency', 'incident_date', 'description']);
    }
}
