<?php

namespace Tests\Feature;

use App\Models\Grievance;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CaseActionsTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_can_view_case_details()
    {
        $admin = User::factory()->create(['email' => 'admin@test.com']);
        $this->actingAs($admin);
        
        $case = Grievance::create([
            'user_id' => $admin->id,
            'title' => 'Test Case',
            'category' => 'other',
            'description' => 'Test Description',
            'incident_date' => now(),
            'urgency' => 'low',
            'status' => 'submitted',
        ]);

        $response = $this->get(route('admin.cases.show', $case->id));
        $response->assertStatus(200);
        $response->assertSee('Test Case');
        $response->assertSee('Test Description');
    }

    public function test_admin_can_update_case()
    {
        $admin = User::factory()->create();
        $this->actingAs($admin);

        $case = Grievance::create([
            'user_id' => $admin->id,
            'title' => 'Old Title',
            'category' => 'other',
            'description' => 'Old Description',
            'incident_date' => now(),
            'urgency' => 'low',
            'status' => 'submitted',
        ]);

        $response = $this->put(route('admin.cases.update', $case->id), [
            'title' => 'New Title',
            'category' => 'wage_dispute',
            'description' => 'New Description',
            'incident_date' => '2026-03-01',
            'urgency' => 'high',
            'status' => 'in_progress',
            'employer_name' => 'New Employer'
        ]);

        $response->assertRedirect(route('admin.cases.index'));
        
        $this->assertDatabaseHas('grievances', [
            'id' => $case->id,
            'title' => 'New Title',
            'urgency' => 'high',
            'status' => 'in_progress',
        ]);
    }

    public function test_admin_can_archive_case()
    {
        $admin = User::factory()->create();
        $this->actingAs($admin);

        $case = Grievance::create([
            'user_id' => $admin->id,
            'title' => 'To Archive',
            'category' => 'other',
            'description' => 'Desc',
            'incident_date' => now(),
            'urgency' => 'low',
            'status' => 'submitted',
        ]);

        $response = $this->post(route('admin.cases.archive', $case->id));
        
        $response->assertRedirect(route('admin.cases.index'));
        
        $this->assertDatabaseHas('grievances', [
            'id' => $case->id,
            'status' => 'archived',
        ]);
    }
}
