<?php

namespace Tests\Feature\Admin;

use App\Models\Branch;
use App\Models\Province;
use App\Models\Region;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OrganizationMasterDataTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
    }

    public function test_admin_can_view_organization_master_data()
    {
        $response = $this->actingAs($this->user)->get(route('admin.organization.index'));
        $response->assertStatus(200);
        $response->assertSee('Organization Master Data');
        $response->assertSee('Provinces');
        $response->assertSee('Regions');
        $response->assertSee('Branches');
    }

    public function test_admin_can_manage_provinces()
    {
        // Store
        $response = $this->actingAs($this->user)->post(route('admin.organization.provinces.store'), [
            'name' => 'New Province',
            'code' => 'NP',
        ]);
        $response->assertRedirect();
        $this->assertDatabaseHas('provinces', ['name' => 'New Province', 'code' => 'NP']);

        $province = Province::where('name', 'New Province')->first();

        // Update
        $response = $this->actingAs($this->user)->put(route('admin.organization.provinces.update', $province), [
            'name' => 'Updated Province',
            'code' => 'UP',
        ]);
        $response->assertRedirect();
        $this->assertDatabaseHas('provinces', ['name' => 'Updated Province']);

        // Destroy
        $response = $this->actingAs($this->user)->delete(route('admin.organization.provinces.destroy', $province));
        $response->assertRedirect();
        $this->assertSoftDeleted('provinces', ['id' => $province->id]); // Oh wait, I didn't add SoftDeletes to models/migration.
        // Checking existing migration/model... I didn't add SoftDeletes trait. So it should be hard deleted.
        $this->assertDatabaseMissing('provinces', ['id' => $province->id]);
    }

    public function test_admin_can_manage_regions()
    {
        $province = Province::create(['name' => 'Test Province', 'code' => 'TP']);

        // Store
        $response = $this->actingAs($this->user)->post(route('admin.organization.regions.store'), [
            'name' => 'New Region',
            'province_id' => $province->id,
        ]);
        $response->assertRedirect();
        $this->assertDatabaseHas('regions', ['name' => 'New Region', 'province_id' => $province->id]);

        $region = Region::where('name', 'New Region')->first();

        // Update
        $response = $this->actingAs($this->user)->put(route('admin.organization.regions.update', $region), [
            'name' => 'Updated Region',
            'province_id' => $province->id,
        ]);
        $response->assertRedirect();
        $this->assertDatabaseHas('regions', ['name' => 'Updated Region']);

        // Destroy
        $response = $this->actingAs($this->user)->delete(route('admin.organization.regions.destroy', $region));
        $response->assertRedirect();
        $this->assertDatabaseMissing('regions', ['id' => $region->id]);
    }

    public function test_admin_can_manage_branches()
    {
        $province = Province::create(['name' => 'Test Province', 'code' => 'TP']);
        $region = Region::create(['name' => 'Test Region', 'province_id' => $province->id]);

        // Store
        $response = $this->actingAs($this->user)->post(route('admin.organization.branches.store'), [
            'name' => 'New Branch',
            'code' => 'NB',
            'region_id' => $region->id,
        ]);
        $response->assertRedirect();
        $this->assertDatabaseHas('branches', ['name' => 'New Branch', 'region_id' => $region->id]);

        $branch = Branch::where('name', 'New Branch')->first();

        // Update
        $response = $this->actingAs($this->user)->put(route('admin.organization.branches.update', $branch), [
            'name' => 'Updated Branch',
            'code' => 'UB',
            'region_id' => $region->id,
        ]);
        $response->assertRedirect();
        $this->assertDatabaseHas('branches', ['name' => 'Updated Branch']);

        // Destroy
        $response = $this->actingAs($this->user)->delete(route('admin.organization.branches.destroy', $branch));
        $response->assertRedirect();
        $this->assertDatabaseMissing('branches', ['id' => $branch->id]);
    }
}
