<?php

namespace Tests\Feature\Admin;

use App\Models\Message;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class MessageArchiveTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create admin user
        $this->admin = User::factory()->create();
        $role = Role::create(['name' => 'admin']);
        $this->admin->assignRole($role);
    }

    public function test_message_can_be_archived()
    {
        $message = Message::create([
            'sender_id' => $this->admin->id,
            'title' => 'Test Message',
            'content' => 'Content',
            'type' => 'announcement',
            'recipient_type' => 'all',
            'status' => 'draft'
        ]);

        $response = $this->actingAs($this->admin)
            ->delete(route('admin.messages.destroy', $message));

        $response->assertRedirect();
        
        // Check if soft deleted
        $this->assertSoftDeleted('messages', ['id' => $message->id]);
    }

    public function test_archived_message_appears_in_archive_list()
    {
        $message = Message::create([
            'sender_id' => $this->admin->id,
            'title' => 'Archived Message',
            'content' => 'Content',
            'type' => 'announcement',
            'recipient_type' => 'all',
            'status' => 'draft'
        ]);

        $message->delete();

        $response = $this->actingAs($this->admin)
            ->get(route('admin.messages.archived'));

        $response->assertStatus(200);
        $response->assertSee('Archived Message');
    }

    public function test_archived_message_can_be_restored()
    {
        $message = Message::create([
            'sender_id' => $this->admin->id,
            'title' => 'Restorable Message',
            'content' => 'Content',
            'type' => 'announcement',
            'recipient_type' => 'all',
            'status' => 'draft'
        ]);

        $message->delete();

        $response = $this->actingAs($this->admin)
            ->post(route('admin.messages.restore', $message->id));

        $response->assertRedirect(route('admin.messages.archived'));
        
        // Check if restored
        $this->assertNotSoftDeleted('messages', ['id' => $message->id]);
    }
}
