<?php

namespace Tests\Feature\Admin;

use App\Models\Event;
use App\Models\EventReminder;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class EventRemindersTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
    }

    public function test_admin_can_add_reminders_to_event()
    {
        $response = $this->actingAs($this->user)->post(route('admin.events.store'), [
            'title' => 'Event with Reminders',
            'event_type' => 'meeting',
            'start_time' => now()->addDays(5)->format('Y-m-d H:i:s'),
            'publish_status' => 'published',
            'reminders' => [
                [
                    'value' => 1,
                    'unit' => 'days',
                    'channels' => ['in_app', 'email']
                ],
                [
                    'value' => 2,
                    'unit' => 'hours',
                    'channels' => ['push']
                ]
            ]
        ]);

        $response->assertRedirect(route('admin.events.index'));
        
        $event = Event::where('title', 'Event with Reminders')->first();
        $this->assertCount(2, $event->reminders);
        
        $this->assertDatabaseHas('event_reminders', [
            'event_id' => $event->id,
            'value' => 1,
            'unit' => 'days'
        ]);
    }

    public function test_reminders_are_processed_correctly()
    {
        $event = Event::create([
            'title' => 'Upcoming Event',
            'event_type' => 'meeting',
            'start_time' => now()->addHour()->addMinutes(10), // Event starts in 70 mins
            'status' => 'upcoming',
            'published_at' => now(),
            'created_by' => $this->user->id
        ]);

        // Reminder set for 2 hours before -> Should NOT be processed yet (Time to send was 50 mins ago? Wait. 
        // If event is in 70 mins. Reminder is 2 hours before.
        // Reminder time = Now + 70m - 120m = Now - 50m.
        // It IS past. So it should run.

        $reminder = $event->reminders()->create([
            'value' => 2,
            'unit' => 'hours',
            'channels' => ['push'],
            'status' => 'pending'
        ]);

        $this->artisan('events:reminders')
             ->assertExitCode(0);

        $this->assertDatabaseHas('event_reminders', [
            'id' => $reminder->id,
            'status' => 'processed'
        ]);
        
        $reminder->refresh();
        $this->assertNotNull($reminder->processed_at);
    }
    
    public function test_future_reminders_are_not_processed()
    {
        $event = Event::create([
            'title' => 'Far Future Event',
            'event_type' => 'meeting',
            'start_time' => now()->addDays(2),
            'status' => 'upcoming',
            'published_at' => now(),
            'created_by' => $this->user->id
        ]);

        // Reminder 1 day before. 
        // Reminder Time = Now + 2 days - 1 day = Now + 1 day.
        // Is NOT past. Should stay pending.

        $reminder = $event->reminders()->create([
            'value' => 1,
            'unit' => 'days',
            'channels' => ['email'],
            'status' => 'pending'
        ]);

        $this->artisan('events:reminders');

        $this->assertDatabaseHas('event_reminders', [
            'id' => $reminder->id,
            'status' => 'pending'
        ]);
    }
}
