<?php

namespace Tests\Feature\Admin;

use App\Models\Event;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class EventPublishingTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
    }

    public function test_admin_can_create_draft_event()
    {
        $response = $this->actingAs($this->user)->post(route('admin.events.store'), [
            'title' => 'Draft Event',
            'event_type' => 'meeting',
            'start_time' => now()->addDay()->format('Y-m-d H:i:s'),
            'publish_status' => 'draft',
        ]);

        $response->assertRedirect(route('admin.events.index'));
        $this->assertDatabaseHas('events', ['title' => 'Draft Event', 'published_at' => null]);
    }

    public function test_admin_can_create_published_event()
    {
        $response = $this->actingAs($this->user)->post(route('admin.events.store'), [
            'title' => 'Published Event',
            'event_type' => 'meeting',
            'start_time' => now()->addDay()->format('Y-m-d H:i:s'),
            'publish_status' => 'published',
        ]);

        $response->assertRedirect(route('admin.events.index'));
        $event = Event::where('title', 'Published Event')->first();
        $this->assertNotNull($event->published_at);
        $this->assertTrue($event->published_at->isPast() || $event->published_at->isToday());
    }

    public function test_admin_can_create_scheduled_event()
    {
        $publishDate = now()->addDays(2);
        
        $response = $this->actingAs($this->user)->post(route('admin.events.store'), [
            'title' => 'Scheduled Event',
            'event_type' => 'meeting',
            'start_time' => now()->addDay()->format('Y-m-d H:i:s'),
            'publish_status' => 'scheduled',
            'published_at' => $publishDate->format('Y-m-d H:i:s'),
        ]);

        $response->assertRedirect(route('admin.events.index'));
        $event = Event::where('title', 'Scheduled Event')->first();
        $this->assertNotNull($event->published_at);
        $this->assertTrue($event->published_at->isFuture());
    }

    public function test_admin_can_update_draft_to_published()
    {
        $event = Event::create([
            'title' => 'Draft Event',
            'event_type' => 'meeting',
            'start_time' => now()->addDay(),
            'published_at' => null // Draft
        ]);

        $response = $this->actingAs($this->user)->put(route('admin.events.update', $event), [
            'title' => 'Published Event',
            'event_type' => 'meeting',
            'start_time' => now()->addDay()->format('Y-m-d H:i:s'),
            'description' => 'Desc',
            'publish_status' => 'published',
        ]);

        $response->assertRedirect(route('admin.events.index'));
        $event->refresh();
        $this->assertNotNull($event->published_at);
    }
}
