<?php

namespace Tests\Feature\Admin;

use App\Models\Event;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class EventManagementTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        // Create a user for authentication
        $this->user = User::factory()->create();
    }

    public function test_admin_can_view_events_index()
    {
        $response = $this->actingAs($this->user)->get(route('admin.events.index'));
        $response->assertStatus(200);
        $response->assertViewIs('admin.events.index');
    }

    public function test_admin_can_create_event()
    {
        $eventData = [
            'title' => 'Test Meeting',
            'event_type' => 'meeting',
            'start_time' => now()->addDay()->format('Y-m-d H:i:s'),
            'end_time' => now()->addDay()->addHour()->format('Y-m-d H:i:s'),
            'location' => 'Board Room',
            'description' => '<p>Meeting Agenda</p>',
            'publish_status' => 'published',
        ];

        $response = $this->actingAs($this->user)->post(route('admin.events.store'), $eventData);

        $response->assertRedirect(route('admin.events.index'));
        $this->assertDatabaseHas('events', [
            'title' => 'Test Meeting',
            'event_type' => 'meeting',
        ]);
    }

    public function test_admin_can_edit_event()
    {
        $event = Event::create([
            'title' => 'Old Title',
            'event_type' => 'meeting',
            'start_time' => now()->addDays(2),
            'status' => 'upcoming'
        ]);

        $updateData = [
            'title' => 'New Title',
            'event_type' => 'conference',
            'start_time' => now()->addDays(3)->format('Y-m-d H:i:s'),
            'description' => 'Updated Description',
            'publish_status' => 'published',
        ];

        $response = $this->actingAs($this->user)->put(route('admin.events.update', $event), $updateData);

        $response->assertRedirect(route('admin.events.index'));
        $this->assertDatabaseHas('events', ['title' => 'New Title']);
    }

    public function test_admin_can_cancel_event()
    {
        $event = Event::create([
            'title' => 'To Be Cancelled',
            'event_type' => 'strike',
            'start_time' => now()->addDays(5),
            'status' => 'upcoming'
        ]);

        $response = $this->actingAs($this->user)->post(route('admin.events.cancel', $event), [
            'reason' => 'Weather conditions'
        ]);

        $response->assertSessionHas('success');
        $this->assertDatabaseHas('events', [
            'id' => $event->id,
            'status' => 'cancelled',
            'cancellation_reason' => 'Weather conditions'
        ]);
    }

    public function test_admin_can_duplicate_event()
    {
        $event = Event::create([
            'title' => 'Original Event',
            'event_type' => 'workshop',
            'start_time' => now()->addDays(10),
            'status' => 'upcoming'
        ]);

        $response = $this->actingAs($this->user)->post(route('admin.events.duplicate', $event));

        $this->assertDatabaseHas('events', [
            'title' => 'Copy of Original Event',
            'event_type' => 'workshop'
        ]);
        
        $newEvent = Event::where('title', 'Copy of Original Event')->first();
        $response->assertRedirect(route('admin.events.edit', $newEvent));
    }

    public function test_admin_can_create_event_with_targeting()
    {
        // Create necessary master data
        $province = \App\Models\Province::create(['name' => 'Western Cape', 'code' => 'WC']);
        $region1 = \App\Models\Region::create(['name' => 'Cape Metro', 'province_id' => $province->id]);
        $region2 = \App\Models\Region::create(['name' => 'Winelands', 'province_id' => $province->id]);

        $eventData = [
            'title' => 'Targeted Event',
            'event_type' => 'meeting',
            'start_time' => now()->addDay()->format('Y-m-d H:i:s'),
            'description' => 'Targeted Description',
            'publish_status' => 'published',
            'targeting' => [
                'type' => 'regions',
                'regions' => [$region1->id, $region2->id], // Send IDs
                'paid_up_only' => true
            ]
        ];

        $response = $this->actingAs($this->user)->post(route('admin.events.store'), $eventData);

        $response->assertRedirect(route('admin.events.index'));
        
        $this->assertDatabaseHas('events', [
            'title' => 'Targeted Event',
        ]);

        $event = Event::where('title', 'Targeted Event')->first();
        $this->assertEquals('regions', $event->targeting['type']);
        // Assert that the stored array matches the sent IDs (as strings if cast but let's check basic equality)
        // Since we send IDs, we expect IDs back.
        // Note: Request params might be strings, so let's be careful with types.
        $this->assertEquals([$region1->id, $region2->id], $event->targeting['regions']);
        $this->assertTrue($event->targeting['paid_up_only']);
    }
}
