<?php

namespace App\Services;

use App\Models\User;
use Carbon\Carbon;

class FeeCalculatorService
{
    public function calculateExpected(User $user)
    {
        $totalExpected = 0;
        $subscriptions = $user->subscriptions()->where('status', 'active')->with('feeStructure')->get();

        foreach ($subscriptions as $sub) {
            $structure = $sub->feeStructure;
            if (!$structure) continue;

            $startDate = Carbon::parse($sub->start_date);
            $endDate = $sub->end_date ? Carbon::parse($sub->end_date) : Carbon::now();
            
            // Limit end date to now if subscription is still active
            if ($endDate->isFuture()) {
                $endDate = Carbon::now();
            }

            if ($structure->frequency === 'one-time') {
                $totalExpected += $structure->amount;
            } elseif ($structure->frequency === 'monthly') {
                $months = $startDate->diffInMonths($endDate) + 1; // +1 to include starting month
                $totalExpected += $months * $structure->amount;
            } elseif ($structure->frequency === 'annual') {
                $years = $startDate->diffInYears($endDate) + 1;
                $totalExpected += $years * $structure->amount;
            }
        }

        return round($totalExpected, 2);
    }

    public function calculateBalance(User $user)
    {
        $expected = $this->calculateExpected($user);
        $paid = $user->payments()->where('status', 'verified')->sum('amount');
        
        return round($expected - $paid, 2); // Positive = Owed (Arrears), Negative/Zero = Paid Up
    }

    public function getStatus(User $user)
    {
        $balance = $this->calculateBalance($user);
        
        if ($balance > 0) {
            return 'Arrears';
        }
        return 'Paid Up';
    }
}
