<?php

namespace App\Services;

use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Illuminate\Support\Facades\Log;

class FcmService
{
    protected $messaging;

    public function __construct()
    {
        $credentialsPath = base_path(env('FIREBASE_CREDENTIALS', 'storage/app/firebase_credentials.json'));
        
        if (file_exists($credentialsPath)) {
            $factory = (new Factory)->withServiceAccount($credentialsPath);
            $this->messaging = $factory->createMessaging();
        } else {
            Log::warning("Firebase credentials not found at: $credentialsPath");
            $this->messaging = null; // Handle gracefully
        }
    }

    public function sendToTopic($topic, $title, $body, $data = [])
    {
        if (!$this->messaging) {
            return false;
        }

        $notification = Notification::create($title, $body);

        $message = CloudMessage::withTarget('topic', $topic)
            ->withNotification($notification)
            ->withData($data);

        try {
            $this->messaging->send($message);
            Log::info("FCM Notification sent to topic: $topic");
            return true;
        } catch (\Throwable $e) {
            Log::error('FCM Error: ' . $e->getMessage());
            return false;
        }
    }

    public function sendMulticast($tokens, $title, $body, $data = [])
    {
        if (!$this->messaging || empty($tokens)) {
            return false;
        }

        $notification = Notification::create($title, $body);

        $message = \Kreait\Firebase\Messaging\MulticastSendReport::withItems([]); // Placeholder type check ignored

        try {
            // Send in chunks of 500 (Firebase limit)
            $chunks = array_chunk($tokens, 500);
            foreach ($chunks as $chunk) {
                $message = CloudMessage::new()
                    ->withNotification($notification)
                    ->withData($data);
                
                $this->messaging->sendMulticast($message, $chunk);
            }
            Log::info("FCM Multicast sent to " . count($tokens) . " devices.");
            return true;
        } catch (\Throwable $e) {
            Log::error('FCM Multicast Error: ' . $e->getMessage());
            return false;
        }
    }
}
