<?php

namespace App\Services;

use App\Models\User;
use App\Models\Grievance;
use App\Models\Fee;
use App\Models\Event;
use App\Models\Message;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardService extends BaseService
{
    /**
     * Get aggregated dashboard statistics.
     *
     * @param array $filters
     * @return array
     */
    public function getStats(array $filters = [])
    {
        $region = $filters['region'] ?? null;
        $department = $filters['department'] ?? null;
        $startDate = $filters['start_date'] ?? Carbon::now()->startOfMonth();
        $endDate = $filters['end_date'] ?? Carbon::now()->endOfMonth();

        // Base queries
        $userQuery = User::query();
        $grievanceQuery = Grievance::query();
        $feeQuery = Fee::query()->whereBetween('date', [$startDate, $endDate]);
        $eventQuery = Event::query()->where('date', '>=', Carbon::now());
        $messageQuery = Message::query()->where('is_read', false);

        // Apply filters
        if ($region) {
            // Assuming users have region, or via relation
             $userQuery->where('region', $region); // Need to verify User model has region, assuming yes for now based on context
             $grievanceQuery->where('region', $region);
             $eventQuery->where(function($q) use ($region) {
                 $q->where('region', $region)->orWhereNull('region');
             });
        }

        if ($department) {
            $grievanceQuery->where('department', $department);
        }

        return [
            'total_members' => $userQuery->count(),
            'active_grievances' => $grievanceQuery->where('status', '!=', 'resolved')->count(),
            'fees_collected' => $feeQuery->where('status', 'paid')->sum('amount'),
            'upcoming_events' => $eventQuery->where('status', 'upcoming')->count(),
            'unread_messages' => $messageQuery->count(),
        ];
    }

    /**
     * Get recent activity across modules.
     *
     * @return array
     */
    public function getRecentActivity()
    {
        $grievances = Grievance::with('user:id,name')->latest()->take(5)->get()->map(function ($item) {
            return [
                'type' => 'grievance',
                'title' => $item->title,
                'user' => $item->user->name,
                'time' => $item->created_at->diffForHumans(),
                'status' => $item->status,
            ];
        });

        $events = Event::latest()->take(3)->get()->map(function ($item) {
            return [
                'type' => 'event',
                'title' => $item->title,
                'time' => $item->date->diffForHumans(),
                'status' => $item->status,
            ];
        });

        return $grievances->merge($events)->sortByDesc('time')->values()->all();
    }

    /**
     * Get pending alerts for admin.
     *
     * @return array
     */
    public function getPendingAlerts()
    {
        return [
            'pending_grievances' => Grievance::where('status', 'pending')->count(),
            'failed_transactions' => Fee::where('status', 'failed')->count(),
        ];
    }
}
