<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Tymon\JWTAuth\Facades\JWTAuth;

use App\Repositories\UserRepository;

class AuthService extends BaseService
{
    protected UserRepository $userRepository;

    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * Register a new user.
     *
     * @param array $data
     * @return array
     */
    public function register(array $data): array
    {
        $user = $this->userRepository->create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
        ]);

        // Assign default role if needed
        // $user->assignRole('user');

        $token = JWTAuth::fromUser($user);

        return [
            'user' => $user,
            'token' => $token,
        ];
    }

    /**
     * Login user and return token.
     *
     * @param array $credentials
     * @return array|null
     */
    public function login(array $credentials): ?array
    {
        if (! $token = Auth::attempt($credentials)) {
            return null;
        }

        $user = Auth::user();

        // Record Login History
        \App\Models\LoginHistory::create([
            'user_id' => $user->id,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);

        return [
            'user' => $user,
            'token' => $token,
        ];
    }

    /**
     * Reset user password.
     *
     * @param User $user
     * @param string $newPassword
     * @return void
     */
    public function resetPassword(User $user, string $newPassword): void
    {
        $user->password = Hash::make($newPassword);
        $user->save();
        
        // Invalidate tokens if necessary
        // JWTAuth::invalidate(true); 
    }

    /**
     * Logout the user (invalidate token).
     *
     * @return void
     */
    public function logout(): void
    {
        Auth::logout();
    }

    /**
     * Refresh the token.
     *
     * @return array
     */
    public function refresh(): array
    {
        return [
            'user' => Auth::user(),
            'token' => Auth::refresh(),
        ];
    }
}
