<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Message extends Model
{
    use \Illuminate\Database\Eloquent\Factories\HasFactory;
    use \Illuminate\Database\Eloquent\SoftDeletes;

    protected $fillable = [
        'sender_id',
        'receiver_id',
        'title',
        'content',
        'type',
        'status',
        'recipient_type',
        'targeting_criteria',
        'published_at',
        'is_read',
        'channels',
        'expires_at',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected $casts = [
        'is_read' => 'boolean',
        'published_at' => 'datetime',
        'expires_at' => 'datetime',
        'targeting_criteria' => 'array',
        'channels' => 'array',
    ];

    /**
     * Scope a query to only include active messages (published and not expired).
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'published')
                     ->whereNotNull('published_at')
                     ->where('published_at', '<=', now())
                     ->where(function ($q) {
                         $q->whereNull('expires_at')
                           ->orWhere('expires_at', '>', now());
                     });
    }

    /**
     * Scope a query to only include scheduled messages.
     */
    public function scopeScheduled($query)
    {
        return $query->where('status', 'scheduled');
    }

    /**
     * Get recipients based on targeting criteria
     */
    public function getRecipientsAttribute()
    {
        if ($this->recipient_type === 'all') {
            return User::role('member')->get(); // Assuming 'member' role exists, or just User::all()
        }

        if ($this->recipient_type === 'individual' && $this->receiver_id) {
            return User::where('id', $this->receiver_id)->get();
        }

        if ($this->recipient_type === 'segment' && $this->targeting_criteria) {
            $query = \App\Models\Member::query();
            $criteria = $this->targeting_criteria;

            if (!empty($criteria['branches'])) {
                $query->whereIn('workplace', $criteria['branches']);
            }
            if (!empty($criteria['regions'])) { // Using postal_province as region proxy for now
                $query->whereIn('postal_province', $criteria['regions']);
            }
            if (!empty($criteria['roles'])) {
                $query->whereIn('designation', $criteria['roles']);
            }
            if (!empty($criteria['status'])) {
                $query->where('status', $criteria['status']); // e.g., 'approved'
            }
             if (!empty($criteria['dues_status'])) {
                $query->where('dues_status', $criteria['dues_status']); // e.g., 'paid'
            }

            // Get associated Users
            return User::whereIn('id', $query->pluck('user_id'))->get();
        }

        return collect();
    }

    public function sender()
    {
        return $this->belongsTo(User::class, 'sender_id');
    }

    public function receiver()
    {
        return $this->belongsTo(User::class, 'receiver_id');
    }

    public function attachments()
    {
        return $this->hasMany(MessageAttachment::class);
    }
}
