<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Event extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'title',
        'description',
        'event_type',
        'start_time',
        'end_time',
        'location',
        'virtual_link',
        'status',
        'cancellation_reason',
        'targeting',
        'created_by',
        'published_at',
    ];

    protected $casts = [
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'targeting' => 'array',
        'published_at' => 'datetime',
    ];

    /**
     * Scope a query to only include upcoming events.
     */
    public function scopeUpcoming($query)
    {
        return $query->where('start_time', '>', now())
                     ->where('status', '!=', 'cancelled')
                     ->orderBy('start_time', 'asc');
    }

    /**
     * Scope a query to only include published events (visible to users).
     */
    public function scopePublished($query)
    {
        return $query->whereNotNull('published_at')
                     ->where('published_at', '<=', now())
                     ->where('status', '!=', 'cancelled');
    }

    /**
     * Scope a query to only include draft events.
     */
    public function scopeDraft($query)
    {
        return $query->whereNull('published_at');
    }

    /**
     * Scope a query to only include scheduled events.
     */
    public function scopeScheduled($query)
    {
        return $query->where('published_at', '>', now());
    }

    /**
     * Accessor to check if event is published.
     */
    public function getIsPublishedAttribute()
    {
        return $this->published_at && $this->published_at->isPast();
    }

    /**
     * Scope a query to only include past events.
     */
    public function scopePast($query)
    {
        return $query->where('start_time', '<=', now())
                     ->orderBy('start_time', 'desc');
    }

    /**
     * Scope a query to only include cancelled events.
     */
    public function scopeCancelled($query)
    {
        return $query->where('status', 'cancelled');
    }

    /**
     * Get the reminders for the event.
     */
    public function reminders()
    {
        return $this->hasMany(EventReminder::class);
    }

    /**
     * Get the user who created the event.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }
}
