<?php

namespace App\Http\Controllers;

use App\Services\FCMService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    protected $fcmService;

    public function __construct(FCMService $fcmService)
    {
        $this->fcmService = $fcmService;
    }

    /**
     * @OA\Post(
     *     path="/api/notifications/send",
     *     summary="Send Push Notification",
     *     tags={"Notifications"},
     *     security={{"bearerAuth":{}}},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"tokens","title","body"},
     *             @OA\Property(property="tokens", type="array", items=@OA\Items(type="string"), description="Device tokens"),
     *             @OA\Property(property="title", type="string", example="Hello"),
     *             @OA\Property(property="body", type="string", example="World"),
     *             @OA\Property(property="data", type="object", description="Additional data")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Notification sent"
     *     )
     * )
     */
    public function send(Request $request): JsonResponse
    {
        $request->validate([
            'tokens' => 'required|array',
            'tokens.*' => 'string',
            'title' => 'required|string',
            'body' => 'required|string',
            'data' => 'nullable|array',
        ]);

        $result = $this->fcmService->sendNotification(
            $request->tokens,
            $request->title,
            $request->body,
            $request->data ?? []
        );

        return response()->json([
            'status' => 'success',
            'message' => 'Notification request processed',
            'data' => $result
        ]);
    }
}
