<?php

namespace App\Http\Controllers;

use App\Models\Member;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;

class MemberController extends Controller
{
    public function index(Request $request)
    {
        $query = Member::query();

        // Apply column filters (regardless of status)
        if ($request->filled('digital_id')) {
            $query->where('digital_id', 'like', "%{$request->digital_id}%");
        }
        if ($request->filled('first_name')) {
            $query->where('first_name', 'like', "%{$request->first_name}%");
        }
        if ($request->filled('last_name')) {
            $query->where('last_name', 'like', "%{$request->last_name}%");
        }
        if ($request->filled('identity_number')) {
            $query->where('identity_number', 'like', "%{$request->identity_number}%");
        }
        if ($request->filled('province')) {
            $prov = $request->province;
            $query->where(function($q) use ($prov) {
                $q->where('home_province', 'like', "%{$prov}%")
                  ->orWhere('postal_province', 'like', "%{$prov}%");
            });
        }

        // Handle Status Filter
        if ($request->status == 'trash') {
            $query->onlyTrashed();
        } elseif ($request->filled('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        } else {
            // "All" selected or no status filter -> Include Trashed members too?
            // User requested "listing all members".
            $query->withTrashed();
        }

        // Handle Dues Filter
        if ($request->filled('dues_status') && $request->dues_status != 'all') {
             $query->where('dues_status', $request->dues_status);
        }

        $members = $query->orderBy('first_name', 'asc')->paginate(10);

        if ($request->ajax()) {
            return response()->json([
                'html' => view('members.partials.table_rows', compact('members'))->render(),
                'pagination' => $members->appends($request->all())->links()->toHtml()
            ]);
        }

        return view('members.index', compact('members'));
    }

    public function create()
    {
        return view('members.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'nullable|email|unique:members', // Made email nullable as per paper form inference
            'identity_number' => 'required|unique:members',
            'gender' => 'required|in:Male,Female',
            'monthly_wages' => 'required|numeric',
            'home_address_line1' => 'required',
            'home_address_line2' => 'required',
            'home_city' => 'required',
            'home_code' => 'required',
            'home_province' => 'required',
            'postal_box' => 'required',
            'postal_city' => 'required',
            'postal_code' => 'required',
            'postal_province' => 'required',
            'recruiter_name' => 'required',
            'agreement_date' => 'required|date',
            'phone' => 'required',
            'workplace' => 'required',
            'status' => 'required|in:pending,approved,rejected,suspended',
            'dues_status' => 'required|in:paid,unpaid,pending',
        ]);

        Member::create($request->all());

        return redirect()->route('members.index')->with('success', 'Member created successfully.');
    }

    public function edit($id)
    {
        $member = Member::withTrashed()->findOrFail($id);
        return view('members.edit', compact('member'));
    }

    public function update(Request $request, $id)
    {
        $member = Member::withTrashed()->findOrFail($id);

        $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'nullable|email|unique:members,email,' . $member->id,
            'identity_number' => 'nullable|unique:members,identity_number,' . $member->id,
            'status' => 'required',
            'dues_status' => 'required',
        ]);

        $member->update($request->all());

        // Auto-restore if status is active/pending and member was deleted
        if ($member->trashed() && in_array($request->status, ['approved', 'pending'])) {
            $member->restore();
            return redirect()->route('members.index')->with('success', 'Member updated and restored successfully.');
        }

        return redirect()->route('members.index')->with('success', 'Member updated successfully.');
    }
    
    public function suspend(Member $member)
    {
        $member->update(['status' => 'suspended']);
        return redirect()->route('members.index')->with('success', 'Member suspended successfully.');
    }

    public function reactivate($id)
    {
        $member = Member::withTrashed()->findOrFail($id);
        
        $member->update(['status' => 'approved']);
        
        if ($member->trashed()) {
            $member->restore();
        }

        return redirect()->route('members.index')->with('success', 'Member re-activated successfully.');
    }

    public function destroy(Member $member)
    {
        $member->delete();
        return redirect()->route('members.index')->with('success', 'Member moved to trash successfully.');
    }

    public function restore($id)
    {
        $member = Member::withTrashed()->findOrFail($id);
        $member->restore();
        return redirect()->route('members.index')->with('success', 'Member restored successfully.');
    }

    public function export()
    {
        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=members_export.csv",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        $columns = array('ID', 'First Name', 'Last Name', 'Email', 'Phone', 'Workplace', 'Status', 'Dues Status', 'Digital ID', 'Joined At');

        $callback = function() use ($columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            Member::chunk(100, function($members) use ($file) {
                foreach ($members as $member) {
                    $row['ID']  = $member->id;
                    $row['First Name'] = $member->first_name;
                    $row['Last Name'] = $member->last_name;
                    $row['Email'] = $member->email;
                    $row['Phone'] = $member->phone;
                    $row['Workplace'] = $member->workplace;
                    $row['Status'] = $member->status;
                    $row['Dues Status'] = $member->dues_status;
                    $row['Digital ID'] = $member->digital_id;
                    $row['Joined At'] = $member->joined_at;

                    fputcsv($file, array($row['ID'], $row['First Name'], $row['Last Name'], $row['Email'], $row['Phone'], $row['Workplace'], $row['Status'], $row['Dues Status'], $row['Digital ID'], $row['Joined At']));
                }
            });
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
