<?php

namespace App\Http\Controllers;

use App\Helpers\EnvironmentManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;

class InstallerController extends Controller
{
    protected $envManager;

    public function __construct(EnvironmentManager $envManager)
    {
        $this->envManager = $envManager;
    }

    public function welcome()
    {
        return view('installer.welcome');
    }

    public function requirements()
    {
        $requirements = [
            'php' => [
                'openssl',
                'pdo',
                'mbstring',
                'tokenizer',
                'xml',
                'ctype',
                'json',
                'bcmath',
            ],
            'apache' => [
                'mod_rewrite',
            ],
        ];

        $results = [];
        $check = true;

        // Check PHP extensions
        foreach ($requirements['php'] as $requirement) {
            $results['requirements'][$requirement] = true;

            if (!extension_loaded($requirement)) {
                $results['requirements'][$requirement] = false;
                $check = false;
            }
        }

        // Simple PHP version check check
        $phpVersion = phpversion();
        $minVersion = '8.2.0';
        $supported = version_compare($phpVersion, $minVersion, '>=');
        
        $results['php'] = [
            'version' => $phpVersion,
            'supported' => $supported,
            'min' => $minVersion
        ];

        if (!$supported) {
            $check = false;
        }

        return view('installer.requirements', compact('results', 'check'));
    }

    public function permissions()
    {
        $permissions = [
            'storage/framework/'     => '775',
            'storage/logs/'          => '775',
            'bootstrap/cache/'       => '775',
        ];

        $results = [];
        $check = true;

        foreach ($permissions as $folder => $permission) {
            $path = base_path($folder);
            $isWritable = is_writable($path);
            
            $results['permissions'][] = [
                'folder' => $folder,
                'permission' => $permission,
                'isSet' => $isWritable,
            ];

            if (!$isWritable) {
                $check = false;
            }
        }

        return view('installer.permissions', compact('results', 'check'));
    }

    public function environment()
    {
        return view('installer.environment');
    }

    public function saveEnvironment(Request $request)
    {
        $rules = [
            'app_name'              => 'required|string|max:50',
            'app_url'               => 'required|url',
            'database_connection'   => 'required|string|max:50',
            'database_host'         => 'required|string|max:50',
            'database_port'         => 'required|numeric',
            'database_name'         => 'required|string|max:50',
            'database_username'     => 'required|string|max:50',
            'database_password'     => 'nullable|string|max:50',
        ];

        $request->validate($rules);

        $results = $this->envManager->saveFileWizard($request->all());

        if ($results) {
            return redirect()->route('installer.overview')
                             ->with(['message' => $results]);
        }

        return redirect()->back()->with(['message' => 'Something went wrong. Please check your settings.']);
    }

    public function overview()
    {
        return view('installer.overview');
    }

    public function install()
    {
        try {
            Artisan::call('migrate:fresh', ['--force' => true]);
            Artisan::call('db:seed', ['--force' => true]);
            Artisan::call('storage:link');
            Artisan::call('key:generate', ['--force' => true]);
        } catch (\Exception $e) {
            return redirect()->back()->with(['message' => $e->getMessage()]);
        }

        return redirect()->route('installer.admin');
    }

    public function adminSetup()
    {
        return view('installer.admin');
    }

    public function createAdmin(Request $request)
    {
        $request->validate([
            'name'     => 'required|string|max:255',
            'email'    => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
        ]);

        try {
            $user = \App\Models\User::create([
                'name'     => $request->name,
                'email'    => $request->email,
                'password' => \Illuminate\Support\Facades\Hash::make($request->password),
            ]);

            // Assign Super Admin Role
            // We assume Spatie Permissions is set up and role 'Super Admin' exists (seeded)
            // If not, we should probably create it or handle it. 
            // For now, let's defensively create if missing or just assign.
            // But usually 'db:seed' should satisfy this.
            
            // Just in case seed didn't create it:
            if (!\Spatie\Permission\Models\Role::where('name', 'Super Admin')->exists()) {
                \Spatie\Permission\Models\Role::create(['name' => 'Super Admin']);
            }
            
            $user->assignRole('Super Admin');
            
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with(['message' => 'Error creating admin: ' . $e->getMessage()]);
        }

        return redirect()->route('installer.finish');
    }

    public function finish()
    {
        $installedLogFile = storage_path('installed');
        $dateStamp = date('Y/m/d h:i:sa');

        if (!file_exists($installedLogFile)) {
            $message = trans('Laravel Application Installer successfully INSTALLED on ') . $dateStamp . "\n";
            file_put_contents($installedLogFile, $message);
        }

        return view('installer.finish');
    }
}
