<?php

namespace App\Http\Controllers;

use App\Http\Requests\FileUploadRequest;
use App\Services\FileService;
use Illuminate\Http\JsonResponse;

class FileController extends Controller
{
    protected $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    /**
     * @OA\Post(
     *     path="/api/files/upload",
     *     summary="Upload a file",
     *     tags={"Files"},
     *     security={{"bearerAuth":{}}},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 @OA\Property(
     *                     property="file",
     *                     type="string",
     *                     format="binary",
     *                     description="File to upload"
     *                 ),
     *                 @OA\Property(
     *                     property="folder",
     *                     type="string",
     *                     description="Folder to store the file in (optional)"
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="File uploaded successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="string", example="success"),
     *             @OA\Property(property="message", type="string", example="File uploaded successfully"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="path", type="string", example="uploads/filename.jpg"),
     *                 @OA\Property(property="url", type="string", example="http://localhost/storage/uploads/filename.jpg")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Validation Error"
     *     )
     * )
     */
    public function upload(FileUploadRequest $request): JsonResponse
    {
        $file = $request->file('file');
        $folder = $request->input('folder', 'uploads');

        $path = $this->fileService->upload($file, $folder);
        // Assuming 'public' disk or similar where we can get a URL. 
        // If 'local', we might need a route to serve it or strictly return path.
        // For now, returning path. 
        
        return response()->json([
            'status' => 'success',
            'message' => 'File uploaded successfully',
            'data' => [
                'path' => $path,
                'url' => url('storage/' . $path) // This assumes 'public' disk linked to storage
            ]
        ]);
    }
}
