<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Member;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class MemberController extends Controller
{
    /**
     * Authenticate Member (Login)
     * Using Identity Number as username.
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'identity_number' => 'required',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        $member = Member::where('identity_number', $request->identity_number)->first();

        // Check if member exists and is active
        if (!$member) {
            return response()->json(['error' => 'Member not found'], 404);
        }

        if ($member->status !== 'approved') {
            return response()->json(['error' => 'Your membership is not active.'], 403);
        }

        // Verify password
        if (!Hash::check($request->password, $member->password)) {
            // IF password is null (first time login), allow them to set it? 
            // OR enforce registration? 
            // For now, let's assume if password is NULL in DB, we treat it as valid match for empty string? 
            // BETTER: Use 'validate' endpoint to check existence, and 'register' to set password.
            // USER REQUEST: "each member profile will be the username... web admin panel must be able to share member profile data"
            // Let's assume standard login. If check fails:
            return response()->json(['error' => 'Invalid credentials'], 401);
        }

        $token = $member->createToken('MobileApp')->plainTextToken;

        return response()->json([
            'message' => 'Login successful',
            'access_token' => $token,
            'token_type' => 'Bearer',
            'member' => $member
        ]);
    }

    /**
     * Validate Member Existence (Pre-login check)
     */
    public function validateMember(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'identity_number' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        $member = Member::where('identity_number', $request->identity_number)->first();

        if (!$member) {
            return response()->json(['found' => false], 404);
        }
        
        // Don't expose sensitive info here, just confirming existence
        $hasPassword = !is_null($member->password);

        return response()->json([
            'found' => true,
            'has_password' => $hasPassword,
            'first_name' => $member->first_name,
            'last_name' => $member->last_name,
            'status' => $member->status
        ]);
    }

    /**
     * Update Profile (Password & Picture)
     */
    public function updateProfile(Request $request)
    {
        $member = $request->user(); // Get authenticated member

        $validator = Validator::make($request->all(), [
            'password' => 'nullable|min:6|confirmed',
            'profile_picture' => 'nullable|image|max:5120', // 5MB max
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        if ($request->hasFile('profile_picture')) {
            // Delete old picture if exists
            if ($member->profile_picture) {
                Storage::delete($member->profile_picture);
            }
            $path = $request->file('profile_picture')->store('profile_pictures', 'public');
            $member->profile_picture = $path;
        }

        if ($request->filled('password')) {
            $member->password = Hash::make($request->password);
        }

        $member->save();

        return response()->json([
            'message' => 'Profile updated successfully',
            'member' => $member
        ]);
    }
}
