<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PressRelease;
use App\Models\PressAttachment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PressReleaseController extends Controller
{
    public function index()
    {
        $releases = PressRelease::orderBy('created_at', 'desc')->paginate(10);
        return view('admin.resources.press.index', compact('releases'));
    }

    public function create()
    {
        return view('admin.resources.press.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
            'status' => 'required|in:draft,published',
            'visibility' => 'required|in:public,members',
            'published_at' => 'nullable|date',
            'attachments.*' => 'nullable|file|mimes:pdf,doc,docx,jpg,jpeg,png|max:10240', // 10MB max
            'is_featured' => 'boolean',
        ]);

        $validated['is_featured'] = $request->has('is_featured');
        if (empty($validated['published_at']) && $validated['status'] === 'published') {
            $validated['published_at'] = now();
        }

        $release = PressRelease::create($validated);

        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('press_attachments', 'public');
                
                PressAttachment::create([
                    'press_release_id' => $release->id,
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        return redirect()->route('admin.resources.press.index')
            ->with('success', 'Press Release created successfully.');
    }

    public function edit(PressRelease $pressRelease) // Route model binding probably needs explicit binding logic if param name differs
    {
        // Using $press to match route resource param 'pres' if generic, but let's stick to explicit model binding
        return view('admin.resources.press.edit', compact('pressRelease'));
    }

    public function update(Request $request, PressRelease $pressRelease)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
            'status' => 'required|in:draft,published',
            'visibility' => 'required|in:public,members',
            'published_at' => 'nullable|date',
            'attachments.*' => 'nullable|file|mimes:pdf,doc,docx,jpg,jpeg,png|max:10240',
            'is_featured' => 'boolean',
        ]);

        $validated['is_featured'] = $request->has('is_featured');
        
        $pressRelease->update($validated);

        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('press_attachments', 'public');
                
                PressAttachment::create([
                    'press_release_id' => $pressRelease->id,
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        return redirect()->route('admin.resources.press.index')
            ->with('success', 'Press Release updated successfully.');
    }

    public function destroy(PressRelease $pressRelease)
    {
        $pressRelease->delete();
        return redirect()->route('admin.resources.press.index')
            ->with('success', 'Press Release deleted successfully.');
    }
}
