<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Message;
use App\Models\MessageAttachment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class MessageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $messages = Message::with('sender')->latest()->paginate(10);
        return view('admin.messages.index', compact('messages'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.messages.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'type' => 'required|in:announcement,alert,reminder,circular,emergency',
            'content' => 'required|string',
            'action' => 'required|in:draft,published',
            'recipient_type' => 'required|in:all,individual,segment',
            'targeting_criteria' => 'nullable|array',
            'attachments.*' => 'nullable|file|max:10240',
            'channels' => 'required|array|min:1',
            'channels.*' => 'in:in_app,push,email',
            'schedule_type' => 'nullable|in:now,later',
            'published_at' => 'nullable|date',
            'expires_at' => 'nullable|date|after:published_at',
        ]);

        $message = new Message($validated);
        
        // Handle Scheduling
        $scheduleType = $request->input('schedule_type', 'now');
        if ($request->action === 'published') {
            if ($scheduleType === 'later' && $request->published_at) {
                $message->status = 'scheduled';
                $message->published_at = $request->published_at;
            } else {
                $message->status = 'published';
                $message->published_at = now();
            }
        } else {
            $message->status = 'draft';
            $message->published_at = null;
        }

        $message->sender_id = auth()->id();
        $message->expires_at = $request->expires_at;
        $message->save();

        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('attachments', 'public');
                
                $message->attachments()->create([
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        // Send Notifications if published immediately
        if ($message->status === 'published') {
            try {
                $fcm = new \App\Services\FcmService();
                $recipients = $message->recipients;
                
                $channels = $message->channels ?? ['in_app'];

                if (in_array('email', $channels)) {
                    foreach ($recipients as $recipient) {
                        \Illuminate\Support\Facades\Log::info("Sending Email to {$recipient->email}: {$message->title}");
                    }
                }

                if (in_array('push', $channels)) {
                     if ($message->recipient_type === 'all') {
                        $fcm->sendToTopic(
                            'all_members', 
                            $message->title, 
                            strip_tags($message->content), 
                            ['message_id' => $message->id, 'type' => $message->type]
                        );
                    } else {
                        $tokens = $recipients->pluck('fcm_token')->filter()->values()->all();
                        
                        if (!empty($tokens)) {
                            $fcm->sendMulticast(
                                $tokens,
                                $message->title,
                                strip_tags($message->content),
                                ['message_id' => $message->id, 'type' => $message->type]
                            );
                        }
                    }
                }

            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error('Notification Error: ' . $e->getMessage());
            }
        }

        $messageText = $message->status === 'published' ? 'Message sent successfully.' : 
                      ($message->status === 'scheduled' ? 'Message scheduled successfully.' : 'Message saved as draft.');

        return redirect()->route('admin.messages.index')
            ->with('success', $messageText);
    }

    public function edit(Message $message)
    {
        return view('admin.messages.edit', compact('message'));
    }

    public function update(Request $request, Message $message)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'type' => 'required|in:announcement,alert,reminder,circular,emergency',
            'content' => 'required|string',
            'action' => 'required|in:draft,published',
            'recipient_type' => 'required|in:all,individual,segment',
            'targeting_criteria' => 'nullable|array',
            'attachments.*' => 'nullable|file|max:10240',
            'channels' => 'required|array|min:1',
            'channels.*' => 'in:in_app,push,email',
            'schedule_type' => 'nullable|in:now,later',
            'published_at' => 'nullable|date',
            'expires_at' => 'nullable|date|after:published_at',
        ]);

        $message->fill($validated);
        
        // Handle Scheduling
        $scheduleType = $request->input('schedule_type', 'now');
        if ($request->action === 'published') {
            if ($scheduleType === 'later' && $request->published_at) {
                $message->status = 'scheduled';
                $message->published_at = $request->published_at;
            } else {
                $message->status = 'published';
                // Only set published_at to now if it was not already published or scheduled in the past that should be sent now
                if (!$message->published_at || $scheduleType === 'now') {
                     $message->published_at = now();
                }
            }
        } else {
            $message->status = 'draft';
        }

        $message->expires_at = $request->expires_at;
        $message->save();

        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('attachments', 'public');
                $message->attachments()->create([
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        // Trigger Notifications if being published (Status changed to published)
        // Note: Logic here simplistically sends if status is published. 
        // In real world, we might want to check if it was ALREADY published to avoid duplicates, unless "Resend" is explicit.
        // For this update action, we assume if user clicks "Update & Resend" (which sends action=published), they want to send it.
        // However, standard update usually shouldn't resend if already published. Valid point often handled by separate Resend action.
        // But if transitioning from Draft -> Published, we MUST send.
        
        if ($message->status === 'published' && $message->wasChanged('status')) {
            try {
                $fcm = new \App\Services\FcmService();
                $recipients = $message->recipients;
                
                $channels = $message->channels ?? ['in_app'];

                if (in_array('email', $channels)) {
                     foreach ($recipients as $recipient) {
                        \Illuminate\Support\Facades\Log::info("Sending Update Email to {$recipient->email}: {$message->title}");
                    }
                }

                if (in_array('push', $channels)) {
                    if ($message->recipient_type === 'all') {
                        $fcm->sendToTopic('all_members', $message->title, strip_tags($message->content), ['message_id' => $message->id, 'type' => $message->type]);
                    } else {
                        $tokens = $recipients->pluck('fcm_token')->filter()->values()->all();
                        if (!empty($tokens)) {
                            $fcm->sendMulticast($tokens, $message->title, strip_tags($message->content), ['message_id' => $message->id, 'type' => $message->type]);
                        }
                    }
                }
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error('Notification Update Error: ' . $e->getMessage());
            }
        }

        return redirect()->route('admin.messages.index')
            ->with('success', 'Message updated successfully.'); 
    }

    public function show(Message $message)
    {
        return view('admin.messages.show', compact('message'));
    }

    public function destroy(Message $message)
    {
        $message->delete();
        return back()->with('success', 'Message archived successfully.');
    }

    public function resend(Message $message)
    {
        if ($message->status !== 'published') {
            return back()->with('error', 'Only published messages can be resent.');
        }

        try {
            $fcm = new \App\Services\FcmService();
            $recipients = $message->recipients;
            
            // This logic mirrors the store method's FCM trigger
            if ($message->recipient_type === 'all') {
                $fcm->sendToTopic(
                    'all_members', 
                    $message->title, 
                    strip_tags($message->content), 
                    ['message_id' => $message->id, 'type' => $message->type]
                );
            } else {
                $tokens = $recipients->pluck('fcm_token')->filter()->values()->all();
                if (!empty($tokens)) {
                    $fcm->sendMulticast(
                        $tokens,
                        $message->title,
                        strip_tags($message->content),
                        ['message_id' => $message->id, 'type' => $message->type]
                    );
                }
            }

            return back()->with('success', 'Message resent successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to resend message: ' . $e->getMessage());
        }
    }

    /**
     * Recall (Unpublish) a message.
     */
    public function recall(Message $message)
    {
        $message->status = 'draft';
        $message->save();

        return back()->with('success', 'Message recalled and set to draft.');
    }

    /**
     * Display a listing of archived messages.
     */
    public function archived()
    {
        $messages = Message::onlyTrashed()->with('sender')->latest()->paginate(10);
        return view('admin.messages.archived', compact('messages'));
    }

    /**
     * Restore the specified archived message.
     */
    public function restore($id)
    {
        $message = Message::onlyTrashed()->findOrFail($id);
        $message->restore();

        return redirect()->route('admin.messages.archived')->with('success', 'Message restored successfully.');
    }
}
