<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\EventReminder;
use Illuminate\Http\Request;

class EventController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $filter = $request->query('filter', 'upcoming');
        
        $query = Event::query();

        if ($filter === 'past') {
            $query->past();
        } elseif ($filter === 'cancelled') {
            $query->cancelled();
        } else {
            $query->upcoming();
        }

        $events = $query->paginate(10);

        return view('admin.events.index', compact('events', 'filter'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $branches = \App\Models\Branch::all();
        $regions = \App\Models\Region::all();
        $provinces = \App\Models\Province::all();
        
        return view('admin.events.create', compact('branches', 'regions', 'provinces'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'event_type' => 'required|string',
            'start_time' => 'required|date',
            'end_time' => 'nullable|date|after:start_time',
            'location' => 'nullable|string',
            'virtual_link' => 'nullable|url',
            'description' => 'nullable|string',
            'targeting' => 'nullable|array',
            'targeting.type' => 'required_with:targeting|in:all,branches,regions,roles,paid_up',
            'targeting.branches' => 'nullable|array',
            'targeting.regions' => 'nullable|array',
            'targeting.roles' => 'nullable|array',
            'targeting.paid_up_only' => 'nullable|boolean',
            'publish_status' => 'required|in:draft,published,scheduled',
            'published_at' => 'nullable|required_if:publish_status,scheduled|date|after:now',
            'notify_now' => 'nullable|boolean',
            'notification_channels' => 'nullable|array',
            'reminders' => 'nullable|array',
            'reminders.*.unit' => 'required|in:days,hours',
            'reminders.*.value' => 'required|integer|min:1',
            'reminders.*.channels' => 'required|array',
        ]);

        $validated['created_by'] = auth()->id();
        $validated['status'] = 'upcoming'; // Default

        // Handle Publishing
        $publishStatus = $request->input('publish_status', 'draft');
        if ($publishStatus === 'published') {
            $validated['published_at'] = now();
        } elseif ($publishStatus === 'scheduled') {
            $validated['published_at'] = $request->input('published_at');
        } else {
            $validated['published_at'] = null; // Draft
        }

        $event = Event::create($validated);

        // Handle Reminders
        if ($request->has('reminders')) {
            foreach ($request->reminders as $reminder) {
                $event->reminders()->create($reminder); // Channels cast automatically
            }
        }

        // Handle Immediate Notification
        if ($request->boolean('notify_now') && $publishStatus === 'published') {
            // Dispatch notification logic (Mocked for now)
            // \App\Jobs\SendEventNotification::dispatch($event, $request->input('notification_channels', []));
            logger()->info("Immediate notification dispatched for event {$event->id} via " . implode(',', $request->input('notification_channels', [])));
        }

        return redirect()->route('admin.events.index')->with('success', 'Event created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Event $event)
    {
        return view('admin.events.show', compact('event'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Event $event)
    {
        $branches = \App\Models\Branch::all();
        $regions = \App\Models\Region::all();
        $provinces = \App\Models\Province::all();

        return view('admin.events.edit', compact('event', 'branches', 'regions', 'provinces'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Event $event)
    {
       $validated = $request->validate([
            'title' => 'required|string|max:255',
            'event_type' => 'required|string',
            'start_time' => 'required|date',
            'end_time' => 'nullable|date|after:start_time',
            'location' => 'nullable|string',
            'virtual_link' => 'nullable|url',
            'description' => 'required|string',
            'targeting' => 'nullable|array',
            'targeting.type' => 'required_with:targeting|in:all,branches,regions,roles,paid_up',
            'targeting.branches' => 'nullable|array',
            'targeting.regions' => 'nullable|array',
            'targeting.roles' => 'nullable|array',
            'targeting.paid_up_only' => 'nullable|boolean',
            'publish_status' => 'nullable|in:draft,published,scheduled',
            'published_at' => 'nullable|required_if:publish_status,scheduled|date',
            'notify_now' => 'nullable|boolean',
            'notification_channels' => 'nullable|array',
            'reminders' => 'nullable|array',
            'reminders.*.unit' => 'required|in:days,hours',
            'reminders.*.value' => 'required|integer|min:1',
            'reminders.*.channels' => 'required|array',
        ]);

        // Handle Publishing
        $publishStatus = $request->input('publish_status');
        if($publishStatus) {
             if ($publishStatus === 'published') {
                $validated['published_at'] = now();
            } elseif ($publishStatus === 'scheduled') {
                $validated['published_at'] = $request->input('published_at');
            } elseif ($publishStatus === 'draft') {
                $validated['published_at'] = null;
            }
        }
        
        $event->update($validated);

        // Handle Reminders (Full sync - delete old, add new)
        $event->reminders()->delete();
        if ($request->has('reminders')) {
            foreach ($request->reminders as $reminder) {
                $event->reminders()->create($reminder);
            }
        }

        // Handle Immediate Notification
        if ($request->boolean('notify_now') && $publishStatus === 'published') {
             logger()->info("Immediate notification dispatched for event {$event->id} via " . implode(',', $request->input('notification_channels', [])));
        }

        return redirect()->route('admin.events.index')->with('success', 'Event updated successfully.'); 
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Event $event)
    {
        $event->delete();
        return back()->with('success', 'Event deleted successfully.');
    }

    public function cancel(Request $request, Event $event)
    {
        $request->validate(['reason' => 'required|string']);
        
        $event->update([
            'status' => 'cancelled',
            'cancellation_reason' => $request->reason
        ]);

        return back()->with('success', 'Event cancelled successfully.');
    }

    public function duplicate(Event $event)
    {
        $newEvent = $event->replicate();
        $newEvent->title = "Copy of " . $event->title;
        $newEvent->status = 'upcoming';
        $newEvent->published_at = null; // Reset to Draft
        $newEvent->created_by = auth()->id();
        $newEvent->save();

        return redirect()->route('admin.events.edit', $newEvent)->with('success', 'Event duplicated. Please update details.');
    }
}
