<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Document;
use App\Models\DocumentCategory;
use App\Models\DocumentVersion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;

class DocumentController extends Controller
{
    public function index(Request $request)
    {
        $query = Document::with(['category', 'latestVersion']);

        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        $documents = $query->orderBy('created_at', 'desc')->paginate(15);
        $categories = DocumentCategory::orderBy('name')->get();

        return view('admin.resources.documents.index', compact('documents', 'categories'));
    }

    public function create()
    {
        $categories = DocumentCategory::orderBy('name')->get();
        return view('admin.resources.documents.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:document_categories,id',
            'description' => 'nullable|string',
            'access_level' => 'required|in:public,members,restricted',
            'allowed_roles' => 'nullable|array',
            'file' => 'required|file|max:20480', // 20MB max
            'version_notes' => 'nullable|string',
        ]);

        $document = Document::create([
            'title' => $validated['title'],
            'category_id' => $validated['category_id'],
            'description' => $validated['description'] ?? null,
            'access_level' => $validated['access_level'],
            'allowed_roles' => $validated['allowed_roles'] ?? [],
            'status' => 'active',
        ]);

        $file = $request->file('file');
        $path = $file->store('documents', 'public');

        DocumentVersion::create([
            'document_id' => $document->id,
            'file_path' => $path,
            'file_name' => $file->getClientOriginalName(),
            'file_size' => $file->getSize(),
            'mime_type' => $file->getClientMimeType(),
            'version_notes' => $validated['version_notes'] ?? 'Initial version',
            'user_id' => Auth::id(),
        ]);

        return redirect()->route('admin.resources.documents.index')->with('success', 'Document uploaded successfully.');
    }

    public function show(Document $document)
    {
        $document->load(['category', 'versions.uploader']);
        return view('admin.resources.documents.show', compact('document'));
    }

    public function edit(Document $document)
    {
        $categories = DocumentCategory::orderBy('name')->get();
        return view('admin.resources.documents.edit', compact('document', 'categories'));
    }

    public function update(Request $request, Document $document)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:document_categories,id',
            'description' => 'nullable|string',
            'access_level' => 'required|in:public,members,restricted',
            'allowed_roles' => 'nullable|array',
            'status' => 'required|in:active,archived',
        ]);

        $document->update($validated);

        return redirect()->route('admin.resources.documents.show', $document)->with('success', 'Document details updated.');
    }

    public function storeVersion(Request $request, Document $document)
    {
        $validated = $request->validate([
            'file' => 'required|file|max:20480',
            'version_notes' => 'required|string',
        ]);

        $file = $request->file('file');
        $path = $file->store('documents', 'public');

        DocumentVersion::create([
            'document_id' => $document->id,
            'file_path' => $path,
            'file_name' => $file->getClientOriginalName(),
            'file_size' => $file->getSize(),
            'mime_type' => $file->getClientMimeType(),
            'version_notes' => $validated['version_notes'],
            'user_id' => Auth::id(),
        ]);

        $document->touch(); // Update updated_at of parent doc

        return redirect()->route('admin.resources.documents.show', $document)->with('success', 'New version uploaded.');
    }

    public function destroy(Document $document)
    {
        $document->delete();
        return redirect()->route('admin.resources.documents.index')->with('success', 'Document moved to trash.');
    }
}
