<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Grievance;
use App\Models\User;
use Illuminate\Http\Request;

class CaseController extends Controller
{
    public function index()
    {
        $cases = Grievance::with('user')
            ->where('status', '!=', 'archived')
            ->orderBy('created_at', 'desc')
            ->paginate(10);
        $title = 'Active Cases';
        return view('admin.cases.index', compact('cases', 'title'));
    }

    public function archived()
    {
        $cases = Grievance::with('user')
            ->where('status', 'archived')
            ->orderBy('updated_at', 'desc')
            ->paginate(10);
        $title = 'Archived Cases';
        return view('admin.cases.index', compact('cases', 'title'));
    }

    public function create()
    {
        try {
            $users = User::role('member')->get();
        } catch (\Spatie\Permission\Exceptions\RoleDoesNotExist $e) {
            $users = User::all();
        }
        
        if ($users->isEmpty()) {
             $users = User::all();
        }
        return view('admin.cases.create', compact('users'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'title' => 'required|string|max:255',
            'category' => 'required|string',
            'description' => 'required|string',
            'incident_date' => 'required|date',
            'urgency' => 'required|in:low,medium,high,critical',
            'employer_name' => 'nullable|string',
        ]);

        $case = Grievance::create($validated);

        return redirect()->route('admin.cases.index')->with('success', 'Case created successfully: ' . $case->reference_number);
    }
    public function show($id)
    {
        $case = Grievance::with('user')->findOrFail($id);
        return view('admin.cases.show', compact('case'));
    }

    public function edit($id)
    {
        $case = Grievance::findOrFail($id);
        return view('admin.cases.edit', compact('case'));
    }

    public function update(Request $request, $id)
    {
        $case = Grievance::findOrFail($id);
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'category' => 'required|string',
            'description' => 'required|string',
            'incident_date' => 'required|date',
            'urgency' => 'required|in:low,medium,high,critical',
            'employer_name' => 'nullable|string',
            'status' => 'required|string',
        ]);
        
        $case->update($validated);
        return redirect()->route('admin.cases.index')->with('success', 'Case updated successfully');
    }

    public function archive($id)
    {
        $case = Grievance::findOrFail($id);
        $case->update(['status' => 'archived']);
        return redirect()->route('admin.cases.index')->with('success', 'Case archived successfully');
    }
}
